%
% Coordinate Descent Algorithm 
%
% Applied to Rosenbrock function
%

clear % forget all variables that were set previously
close % close the current figure 
clc   % clear the command history

%%%% ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ %%%%
%%%%                       Function definitions                        %%%%
%%%% ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ %%%%

function f = fun(x)
    %
    % Rosenbrock function 
    %
    % INPUT:
    %   x, [2,n]
    % OUTPUT:
    %   f, [1,n]
    %
    
    f = 100 *(x(2,:) - x(1,:).^2).^2 + (1-x(1,:)).^2;
    
end

%%%% ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ %%%%
%%%%                           Main Script                             %%%%
%%%% ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ %%%%

%%% Create a plot
% create a grid of all combinations of the given values for x1 and x2
[X1,X2] = meshgrid( linspace(-2,2,100), linspace(-1.5,2.5,100));
sizeMG = size(X2);

% we need to flatten the matrices X1,X2 and stack them to a 2xN matrix, 
% such that it can be processed by fun
F = fun([reshape(X1,1,[]); reshape(X2,1,[]) ]);

% F comes back as row vector and must be transformed to the same size as 
% X1,X2
F = reshape(F, sizeMG);


figure   % open a new figure 

hold on   % allow adding new plots to the figure
contour(X1,X2,F,linspace(0,500,30))   % draw the levels of our function 
                                      % (4. argument defines the levels)
colorbar 


%%% Plot start point and minimizer
xOpt = [1;1];
plot(xOpt(1), xOpt(2), 'kx', 'markersize', 10, 'linewidth', 2)

x0 = [-1.9; 2];
plot(x0(1), x0(2), 'ko', 'markersize', 10, 'linewidth', 2)


%%% Start Optimization
h = 0.5;  % Initial step width
hMin = 1e-5;   % Minimum step width allowed
kMax = 1000;   % Maximum number of iterations allowed

x = x0;
f = fun(x);
for k = 1:kMax
    
    xNext = x;
    %%% Loop over dimensions
    for i = 1:2 
        
        s = [0;0];  % unit step in the current dimension
        s(i) = 1;

        fR = fun(xNext+h*s); % evaluate step to the right and to the left
        fL = fun(xNext-h*s);
        
        [f,idx] = min([fL, f, fR]); % which direction is best?
        if idx == 1
            xNext = xNext-h*s;
        elseif idx == 3
            xNext = xNext+h*s;
        end
        
        %%% Plot step
        plot([x(1), xNext(1)], [x(2), xNext(2)], 'k-')
        drawnow
        
    end
    
    %%% Print progress
    if ~mod(k-1,15) % print header every 15 iterations
        fprintf('\n')
        fprintf('   k         h     x1     x2         f\n')
        fprintf('--------------------------------------\n')
        %        ++++ +++++++++ ++++++ ++++++ +++++++++
    end
    fprintf('%4i %9.2e %6.3f %6.3f %9.2e\n', k, h, x(1), x(2), f)
    
    %%% Adjust step size if we did not make any progress
    if all(xNext == x)
        hNew = max(h/2, hMin);
        
        % Terminate if there is no change anymore
        if hNew == h
            x = xNext;
            fprintf('Done: x = [%.2f, %.2f]\n', x(1), x(2))
            break
        end
        
        h = hNew;
    end
    
    x = xNext;
    
end