%
% H09.02
%
% Calculate the stable position of a chain of length L consisting of n segments 
% The ends of the chain are attached to two fixed points (x0,y0) and (xn,yn).
% 
% We use a state vector z: [x1,y1, ... x(n-1),y(n-1), lam1, ... lam(n)]'
% and a parameter-struct para that holds the constants.
% 
% Constraints and objective are defined in separate functions. 
% These are used to build the KKT-system. 
% Finally the system is solved using Newton's method.
%

clear
close
clc

function x = newton(fun,x0)     
    % x = newton(fun,x0)     
    % 
    % Newton algorithm for solving the equation fun(x) = 0
    %
    % INPUT
    %   fun
    %     function handle: [y,dy] = fun(x)
    %     Objective function
    %   x0 
    %     [n,1]
    %     Initial guess
    % OUTPUT:
    %   x
    %     [n,1]
    %     Solution
    %
    
    % ...
    % Implement Newton's method
    % ...
    
end     

function [c,cGrad,cHess] = constrFun(z,para)
    % [c,cGrad,cHess] = constrFun(z,para)
    % 
    % Set up the necessary equality constraints.
    % 
    % INPUT:
    %   z
    %     [2*(n-1)+n, 1]
    %     State vector
    %   para
    %     struct
    %     .L 
    %       scalar
    %       chain length
    %     .n
    %       scalar
    %       Number of chain segments
    % OUTPUT:
    %   c
    %     [n,1]
    %     One constraint for the each chain-segment, that fixes the distance of 
    %     the previous joint.
    %   cGrad
    %     [(n-1)*2, n]
    %     Gradient (Jacobian)
    %   cHess
    %     [(n-1)*2, (n-1)*2, n]
    %     Hessian
    % 
    % Note:
    % This function does not need the Lagrange-Multipliers from z. Only for 
    % simplicity we pass the full state vector. The gradient and hessian are 
    % calculated only w.r.t. the x and y-components
    %
    
    %%% Extract input
    n = para.n; % Number of segments
    [x,y,~] = extractState(z,para);
    
    %%% Set up equality constraints for each chain segment
    c = zeros(n,1);
    cGrad = zeros((n-1)*2, n); % derivative w.r.t x,y
    cHess = zeros((n-1)*2, (n-1)*2, n);
    
    % ...
    % Calculate the constraints, jacobian and hessian
    % ...
    
end

function [f,fGrad,fHess] = objFun(z,para)
    % [f,fGrad,fHess] = objFun(z,para)
    % 
    % Objective function. The potential energy of all segments.
    % 
    % INPUT:
    %   z
    %     [2*(n-1)+n, 1]
    %     State vector
    %   para
    %     struct
    %     .n
    %       scalar
    %       Number of chain segments
    % OUTPUT:
    %   f
    %     scalar
    %     Objective
    %   fGrad
    %     [(n-1)*2, 1]
    %     Gradient
    %   fHess
    %     [(n-1)*2, (n-1)*2, 1]
    %     Hessian
    %
    % Note:
    % This function does not need the Lagrange-Multipliers from z. Only for 
    % simplicity we pass the full state vector. The gradient and hessian are 
    % calculated only w.r.t. the x and y-components
    %
    
    n = para.n;
    [x,y] = extractState(z,para);
    
    % ...
    % Calculate the objective function, gradient and hessian
    % ...
    
end

function [x,y,lam] = extractState(z,para)
    % [x,y,lam] = extractState(z,para)
    %
    % Extracts the state vector and returns the content in a more readable way.
    % 
    % INPUT:
    %   z
    %     [2*(n-1)+n, 1]
    %     State vector
    %   para
    %     struct
    %     .n
    %       scalar
    %       Number of chain segments
    %     .x0, .y0, .xn, .yn
    %       scalar
    %       Coordinates of the start- and end-point, respectively
    % OUTPUT:
    %   x, y
    %     [n+1,1]
    %     x- and y-coordinates including start- and end-point.
    %   lam
    %     [n,1]
    %     Lagrange-Multipliers for the n constraints.
    %
    
    n = para.n; % Number of segments
    
    x = [para.x0; z(1:2:2*(n-1)); para.xn]; % [n+1, 1]
    y = [para.y0; z(2:2:2*(n-1)); para.yn];
    lam = z(2*(n-1)+1:end);  % [n,1]
        
end

function z = compressState(x,y,lam,para)
    % z = CompressState(x,y,lam,para)
    %
    % Combines the coordinates and Lagrange-Multipliers to a single state vector
    % 
    % INPUT:
    %   x, y
    %     [n+1,1]
    %     x- and y-coordinates including start- and end-point.
    %   lam
    %     [n,1]
    %     Lagrange-Multipliers for the n constraints.
    %   para
    %     struct
    %     .n
    %       scalar
    %       Number of chain segments
    % OUTPUT:
    %   z
    %     [2*(n-1)+n, 1]
    %     State vector
    %
    
    n = para.n; % Number of segments
    
    z = zeros(2*(n-1)+n,1);
    
    z(1:2:2*(n-1)) = x(2:end-1);
    z(2:2:2*(n-1)) = y(2:end-1);
    z(2*(n-1)+1:end) = lam;
        
end

function [kkt, dkkt] = KKT(z, para)
    % [kkt, dkkt] = KKT(z, para)
    %
    % Sets up the KKT-system and its gradient. 
    % 
    %
    % INPUT:
    %   z
    %     [(n-1)*2 + n, 1]
    %     State vector
    %   para
    %     struct
    %     Problem parameters.
    % OUTPUT:
    %   kkt
    %     [(n-1)*2 + n, 1]
    %     KKT-system 
    %   dkkt
    %     [(n-1)*2 + n, (n-1)*2 + n,]
    %     Gradient (Jacobian) wrt the state vector
    %
    
    % Extract input
    [x,y,lam] = extractState(z,para);
    
    % Objective function 
    [~,fGrad,fHess] = objFun(z,para);
    
    % Constraints
    [c,cGrad,cHess] = constrFun(z,para);
        
        
    %%% Build KKT system
    
    % ...
    % Build the KKT system and its jacobian
    % ...
    
end
    
%%%% ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ %%%%    
%%%%                          Main Script                                   %%%%
%%%% ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ %%%%    

%%% Set parameters
para.x0 = 0; % Initial point
para.y0 = 0;
para.xn = 1; % End point
para.yn = 1; 
para.L = 1;  % Chain-length 
para.n = 7;  % Number of chain segments

%%% Create initial guess

% ...

%%% Solve the problem

% ...

%%% Plot and validate the solution

% ...